/* Copyright (c) 2001-2008, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.display;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*; 
import java.awt.color.*; 
import java.awt.geom.*; 
import java.util.*; 
import java.io.*; 
import javax.swing.*; 
import javax.swing.event.*;

import com.pixelmed.display.event.*; 
import com.pixelmed.dicom.*;
import com.pixelmed.event.EventContext;

/**
 * <p>This class is an entire application for displaying and viewing chest x-ray images.</p>
 * 
 * 
 * <p>It is invoked using a main method with a list of DICOM image file names.</p>
 * 
 * @author	dclunie
 */
public class ChestImageViewer {

	private static final String identString = "@(#) $Header: $";

	protected JFrame frame;
	protected JPanel multiPanel;
	protected int frameWidth;
	protected int frameHeight;
	
	/**
	 * @param	list
	 * @return			array of two String values, row then column orientation, else array of two nulls if cannot be obtained
	 */
	private static String[] getPatientOrientation(AttributeList list) {
		String[] vPatientOrientation = null;
		Attribute aPatientOrientation = list.get(TagFromName.PatientOrientation);
		if (aPatientOrientation != null && aPatientOrientation.getVM() == 2) {
			try {
				vPatientOrientation = aPatientOrientation.getStringValues();
				if (vPatientOrientation != null && vPatientOrientation.length != 2) {
					vPatientOrientation=null;
				}
			}
			catch (DicomException e) {
				vPatientOrientation=null;
			}
		}
		if (vPatientOrientation == null) {
			vPatientOrientation = new String[2];
		}
		return vPatientOrientation;
	}
	
	/**
	 * @param	list
	 * @return			a single String value, null if cannot be obtained
	 */
	private static String getView(AttributeList list) {
		String view = null;
		CodedSequenceItem csiViewCodeSequence = CodedSequenceItem.getSingleCodedSequenceItemOrNull(list,TagFromName.ViewCodeSequence);
		if (csiViewCodeSequence != null) {
			//view = decipherSNOMEDCodeSequence(csiViewCodeSequence,standardViewCodes);
			view = MammoDemographicAndTechniqueAnnotations.getViewAbbreviationFromViewCodeSequenceAttributes(csiViewCodeSequence.getAttributeList());
		}
//System.err.println("getView(): view="+view);
		return view;
	}
	
	/**
	 * @param	list
	 * @return			a single String value, null if cannot be obtained
	 */
	private static String getImageLateralityViewModifierAndViewModifier(AttributeList list) {
		return MammoDemographicAndTechniqueAnnotations.getAbbreviationFromImageLateralityViewModifierAndViewModifierCodeSequenceAttributes(list);
	}
	
	/**
	 * @param	list
	 * @return			a single String value, null if cannot be obtained
	 */
	private static String getLaterality(AttributeList list) {
		return Attribute.getSingleStringValueOrNull(list,TagFromName.ImageLaterality);
	}
	
	/**
	 * @param	list
	 * @return			a single String value, null if cannot be obtained
	 */
	private static String getDate(AttributeList list) {
		return Attribute.getSingleStringValueOrNull(list,TagFromName.StudyDate);
	}
	
	/**
	 * @param	dates
	 * @return			a single String value, null if cannot be obtained
	 */
	private static String getDateOfCurrentStudy(String dates[]) {
		int n = dates.length;
		int latestDateValue = 0;
		String latestDateString = null;
		for (int i=0; i<n; ++i) {
			if (dates[i] != null) {
				int testDateValue = 0;
				try {
					testDateValue = Integer.parseInt(dates[i]);
				}
				catch (NumberFormatException e) {
					e.printStackTrace(System.err);
				}
				if (testDateValue > latestDateValue) {
					latestDateValue = testDateValue;
					latestDateString = dates[i];
				}
			}
		}
		return latestDateString;
	}
	
	class OurSingleImagePanel extends SingleImagePanelWithRegionDrawing /* SingleImagePanelWithLineDrawing */  {
		public OurSingleImagePanel(SourceImage sImg,EventContext typeOfPanelEventContext) {
			super(sImg,typeOfPanelEventContext);
		}
		protected Shape makeNewDrawingShape(int tlhcX,int tlhcY,int width,int height) {
			return new Ellipse2D.Double(tlhcX,tlhcY,width,height);
		}
	}
	
	protected SingleImagePanel makeNewImagePanel(SourceImage sImg,EventContext typeOfPanelEventContext) {
		return new OurSingleImagePanel(sImg,typeOfPanelEventContext);
	}
	
	/**
	 * @param		filenames
	 * @exception	Exception		if internal error
	 */
	public void loadMultiPanelFromSpecifiedFiles(String filenames[]) throws Exception {
	
		int nFiles = filenames.length;
		
		SingleImagePanel imagePanels[] = new SingleImagePanel[nFiles];
		
		String orientations[][] = new String[nFiles][];
		String views[] = new String[nFiles];
		String lateralityViewAndModifiers[] = new String[nFiles];
		String lateralities[] = new String[nFiles];
		String dates[] = new String[nFiles];
		int widths[] = new int[nFiles];
		int heights[] = new int[nFiles];
		
		String rowOrientations[] = new String[nFiles];
		String columnOrientations[] = new String[nFiles];
		
		HashMap eventContexts = new HashMap();
		
		double maximumHorizontalExtentInMm = 0;
		double maximumVerticalExtentInMm = 0;
		
		StructuredReport sr[] = new StructuredReport[nFiles];

		int nImages = 0;
		int nCAD = 0;
		ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
		for (int f=0; f<nFiles; ++f) {
			try {
				String filename = filenames[f];
				DicomInputStream distream = null;
				InputStream in = classLoader.getResourceAsStream(filename);
				if (in != null) {
					distream = new DicomInputStream(in);
				}
				else {
					distream = new DicomInputStream(new File(filename));
				}
				AttributeList list = new AttributeList();
				list.read(distream);
				if (list.isImage()) {
					int i = nImages++;
System.err.println("IMAGE ["+i+"] is file "+f+" ("+filenames[f]+")");

					orientations[i] = getPatientOrientation(list);
//System.err.println("IMAGE ["+i+"] orientation="+(orientations[i] == null && orientations[i].length == 2 ? "" : (orientations[i][0] + " " + orientations[i][1])));
					views[i] = getView(list);
//System.err.println("IMAGE ["+i+"] view="+views[i]);
					lateralityViewAndModifiers[i] = getImageLateralityViewModifierAndViewModifier(list);
//System.err.println("IMAGE ["+i+"] lateralityViewAndModifiers="+lateralityViewAndModifiers[i]);
//System.err.println("File "+filenames[f]+": "+lateralityViewAndModifiers[i]);
					lateralities[i] = getLaterality(list);
//System.err.println("IMAGE ["+i+"] laterality="+lateralities[i]);
					dates[i] = getDate(list);
//System.err.println("IMAGE ["+i+"] date="+dates[i]);
				
					SourceImage sImg = new SourceImage(list);
					BufferedImage img = sImg.getBufferedImage();
				
					widths[i] = sImg.getWidth();
					heights[i] = sImg.getHeight();
				
					boolean shareVOIEventsInStudy = false;		// does not seem to work anyway, since adding VOITransform to panel constructor :(
				
					EventContext eventContext = null;
					if (shareVOIEventsInStudy) {
						eventContext = (EventContext)(eventContexts.get(dates[i]));	// use same context for studies with same date
						if (eventContext == null) {
							eventContext = new EventContext(dates[i]);
							eventContexts.put(dates[i],eventContext);
						}
					}
					if (eventContext == null) {
						eventContext = new EventContext(Integer.toString(i));
					}
				
					SingleImagePanel imagePanel = makeNewImagePanel(sImg,eventContext);
					imagePanel.setDemographicAndTechniqueAnnotations(new DemographicAndTechniqueAnnotations(list),"SansSerif",Font.PLAIN,10,Color.pink);
					imagePanel.setOrientationAnnotations(
						new OrientationAnnotations(rowOrientations[i],columnOrientations[i]),
						"SansSerif",Font.PLAIN,20,Color.pink);
					if (Attribute.getSingleStringValueOrEmptyString(list,TagFromName.VOILUTFunction).equals("SIGMOID")) {
						imagePanel.setVOIFunctionToLogistic();
					}
					imagePanels[i] = imagePanel;
				}
				else {
					throw new DicomException("Unsupported SOP Class in file "+filenames[f]);
				}
			}
			catch (Exception e) {	// FileNotFoundException,IOException,DicomException
				e.printStackTrace(System.err);
			}
		}

		int imagesPerRow = nImages;			// i.e., 1 -> 1, 2 -> 1, 4 -> 4, 5 -> 4, 8 -> 4
		int imagesPerCol = 1;

		int singleWidth  = frameWidth /imagesPerRow;
		int singleHeight = frameHeight/imagesPerCol;

		if (nImages == 1 && singleWidth > singleHeight) {
			singleWidth = singleWidth/2;			// use only half the screen for a single view and a landscape monitor
		}

		for (int i=0; i<nImages; ++i) {
			DisplayedAreaSelection displayedAreaSelection = null;
			displayedAreaSelection = new DisplayedAreaSelection(widths[i],heights[i],0,0,widths[i],heights[i],
					true,	// in case spacing was not supplied
					0,0,0,0,0,false/*crop*/);
			imagePanels[i].setDisplayedAreaSelection(displayedAreaSelection);
			imagePanels[i].setPreTransformImageRelativeCoordinates(null);
		}
		
		SingleImagePanel.deconstructAllSingleImagePanelsInContainer(multiPanel);
		multiPanel.removeAll();
		multiPanel.setLayout(new GridLayout(imagesPerCol,imagesPerRow));
		multiPanel.setBackground(Color.black);
		
		for (int x=0; x<imagesPerCol; ++x) {
			for (int y=0; y<imagesPerRow; ++y) {
				int i = x*imagesPerRow+y;
				if (i < nImages) {
					imagePanels[i].setPreferredSize(new Dimension(singleWidth,singleHeight));
					multiPanel.add(imagePanels[i]);
				}
			}
		}
		frame.getContentPane().validate();
		frame.getContentPane().repaint();
	}
	
	/**
	 * @param		filenames
	 * @exception	Exception		if internal error
	 */
	public ChestImageViewer(String filenames[]) throws Exception {
		java.awt.GraphicsDevice[] gs = java.awt.GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices();
System.err.println("ChestImageViewer(): gs.length="+gs.length);
		//java.awt.GraphicsDevice gd = gs[gs.length-1];
		java.awt.GraphicsDevice gd = gs[0];
System.err.println("ChestImageViewer(): selected gd="+gd);

		frame = new JFrame(gd.getDefaultConfiguration());

		try {
			Class classToUse = new java.net.URLClassLoader(new java.net.URL[]{new File("/System/Library/Java").toURL()}).loadClass("com.apple.cocoa.application.NSMenu");
			Class[] parameterTypes = { Boolean.TYPE };
			java.lang.reflect.Method methodToUse = classToUse.getDeclaredMethod("setMenuBarVisible",parameterTypes);
			Object[] args = { Boolean.FALSE };
			methodToUse.invoke(null/*since static*/,args);
		}
		catch (Exception e) {	// ClassNotFoundException,NoSuchMethodException,IllegalAccessException
			//e.printStackTrace(System.err);
		}

		frame.setUndecorated(true);
		frame.setLocation(0,0);
		frame.setVisible(true);

		Rectangle d = gd.getDefaultConfiguration().getBounds();
		int frameWidth  = (int)d.getWidth();
		int frameHeight = (int)d.getHeight();
//System.err.println("frameWidth="+frameWidth);
//System.err.println("frameHeight="+frameHeight);
		frame.setSize(frameWidth,frameHeight);
		doCommonConstructorStuff();
		loadMultiPanelFromSpecifiedFiles(filenames);
	}
	
	/**
	 * @exception	Exception		if internal error
	 */
	protected void doCommonConstructorStuff() throws Exception {
		Container content = frame.getContentPane();
		content.setLayout(new GridLayout(1,1));
		multiPanel = new JPanel();
		//multiPanel.setBackground(Color.black);
		frameWidth  = (int)frame.getWidth();
		frameHeight = (int)frame.getHeight();
		Dimension d = new Dimension(frameWidth,frameHeight);
		//multiPanel.setSize(d);
		multiPanel.setPreferredSize(d);
		multiPanel.setBackground(Color.black);
		content.add(multiPanel);
		//frame.pack();
		content.validate();
	}
	
	/**
	 */
	public void clear() {
		SingleImagePanel.deconstructAllSingleImagePanelsInContainer(multiPanel);
		multiPanel.removeAll();
		frame.getContentPane().validate();
		frame.getContentPane().repaint();
	}

	/**
	 * <p>The method to invoke the application.</p>
	 *
	 * @param	arg		a list of DICOM files which may contain chest x-ray images
	 */
	public static void main(String arg[]) {
		try {
			new ChestImageViewer(arg);
		}
		catch (Exception e) {
			e.printStackTrace(System.err);
		}
	}
}

